<?php
/**
 * Script PHP para detectar diferenças entre dois bancos de dados MySQL.
 * Compara: 
 * - Tabelas existentes
 * - Estrutura das tabelas (colunas, tipos, chaves primárias, índices)
 * - Contagem de registros (para diferenças de dados básicas)
 * 
 * Configurações:
 * - Servidor: localhost (ajuste se necessário)
 * - Usuário e senha: defina abaixo
 * - Bancos: linkvistcom_homologacao e linkvistcom_VISTORIA
 * 
 * Aviso: Este script não compara dados linha a linha (seria muito pesado). 
 * Para isso, use ferramentas como mysqldump ou scripts mais avançados.
 * 
 * Execute via CLI: php script.php
 */

// Configurações de conexão
$host = 'localhost';
$username = 'linkvistcom_root'; // Substitua pelo usuário do MySQL
$password = '0Ft%=x6rPJl=';   // Substitua pela senha
$db1 = 'linkvistcom_homologacao';
$db2 = 'linkvistcom_VISTORIA';

try {
    // Conexão PDO para consultas gerais
    $pdo = new PDO("mysql:host=$host", $username, $password, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8"
    ]);

    echo "=== DETECÇÃO DE DIFERENÇAS ENTRE BANCOS: $db1 vs $db2 ===\n\n";

    // 1. Comparar listas de tabelas
    echo "1. DIFERENÇAS EM TABELAS:\n";
    $tables1 = getTables($pdo, $db1);
    $tables2 = getTables($pdo, $db2);
    
    $onlyInDb1 = array_diff($tables1, $tables2);
    $onlyInDb2 = array_diff($tables2, $tables1);
    
    if (!empty($onlyInDb1)) {
        echo "   - Tabelas apenas em $db1: " . implode(', ', $onlyInDb1) . "\n";
    }
    if (!empty($onlyInDb2)) {
        echo "   - Tabelas apenas em $db2: " . implode(', ', $onlyInDb2) . "\n";
    }
    if (empty($onlyInDb1) && empty($onlyInDb2)) {
        echo "   - Todas as tabelas são iguais.\n";
    }
    echo "\n";

    // 2. Comparar estruturas das tabelas comuns
    $commonTables = array_intersect($tables1, $tables2);
    foreach ($commonTables as $table) {
        echo "2. DIFERENÇAS NA TABELA: $table\n";
        $diff = compareTableStructure($pdo, $db1, $db2, $table);
        if (!empty($diff)) {
            echo "   " . implode("\n   ", $diff) . "\n";
        } else {
            echo "   - Estrutura idêntica.\n";
        }
        echo "\n";
    }

    // 3. Comparar contagem de registros nas tabelas comuns
    echo "3. DIFERENÇAS EM CONTAGEM DE REGISTROS:\n";
    foreach ($commonTables as $table) {
        $count1 = getRowCount($pdo, $db1, $table);
        $count2 = getRowCount($pdo, $db2, $table);
        if ($count1 != $count2) {
            echo "   - $table: $db1 ($count1) vs $db2 ($count2)\n";
        }
    }
    if (empty($commonTables) || /* se todas iguais, mas pra simplicidade */ true) {
        // Não imprime nada se todas iguais, mas o loop cuida
    }
    echo "\n";

    echo "=== FIM DA ANÁLISE ===\n";

} catch (PDOException $e) {
    echo "Erro de conexão: " . $e->getMessage() . "\n";
}

/**
 * Função para obter lista de tabelas de um banco
 */
function getTables($pdo, $database) {
    $stmt = $pdo->prepare("
        SELECT TABLE_NAME 
        FROM INFORMATION_SCHEMA.TABLES 
        WHERE TABLE_SCHEMA = ? AND TABLE_TYPE = 'BASE TABLE'
        ORDER BY TABLE_NAME
    ");
    $stmt->execute([$database]);
    return $stmt->fetchAll(PDO::FETCH_COLUMN);
}

/**
 * Função para comparar estrutura de uma tabela
 * Retorna array de strings com diferenças
 */
function compareTableStructure($pdo, $db1, $db2, $table) {
    $diffs = [];

    // Comparar colunas
    $cols1 = getColumns($pdo, $db1, $table);
    $cols2 = getColumns($pdo, $db2, $table);
    
    $onlyIn1 = array_diff_assoc($cols1, $cols2);
    $onlyIn2 = array_diff_assoc($cols2, $cols1);
    
    foreach ($onlyIn1 as $col => $info) {
        $diffs[] = "Coluna '$col' existe apenas em $db1 (tipo: {$info['type']})";
    }
    foreach ($onlyIn2 as $col => $info) {
        $diffs[] = "Coluna '$col' existe apenas em $db2 (tipo: {$info['type']})";
    }
    
    // Comparar diferenças em colunas comuns
    $commonCols = array_intersect(array_keys($cols1), array_keys($cols2));
    foreach ($commonCols as $col) {
        if ($cols1[$col]['type'] != $cols2[$col]['type'] || 
            $cols1[$col]['nullable'] != $cols2[$col]['nullable'] ||
            $cols1[$col]['default'] != $cols2[$col]['default']) {
            $diffs[] = "Coluna '$col': tipo={$cols1[$col]['type']}->{$cols2[$col]['type']}, nullável={$cols1[$col]['nullable']}->{$cols2[$col]['nullable']}, default={$cols1[$col]['default']}->{$cols2[$col]['default']}";
        }
    }

    // Comparar chaves primárias
    $pk1 = getPrimaryKey($pdo, $db1, $table);
    $pk2 = getPrimaryKey($pdo, $db2, $table);
    if ($pk1 != $pk2) {
        $diffs[] = "Chave primária: " . implode(',', $pk1) . " vs " . implode(',', $pk2);
    }

    // Comparar índices (simples, sem chaves estrangeiras por simplicidade)
    $idx1 = getIndexes($pdo, $db1, $table);
    $idx2 = getIndexes($pdo, $db2, $table);
    $onlyIdx1 = array_diff($idx1, $idx2);
    $onlyIdx2 = array_diff($idx2, $idx1);
    if (!empty($onlyIdx1)) {
        $diffs[] = "Índices apenas em $db1: " . implode(', ', $onlyIdx1);
    }
    if (!empty($onlyIdx2)) {
        $diffs[] = "Índices apenas em $db2: " . implode(', ', $onlyIdx2);
    }

    return $diffs;
}

/**
 * Obter colunas de uma tabela (nome => info)
 */
function getColumns($pdo, $database, $table) {
    $stmt = $pdo->prepare("
        SELECT 
            COLUMN_NAME as col,
            DATA_TYPE as type,
            IS_NULLABLE as nullable,
            COLUMN_DEFAULT as default_val
        FROM INFORMATION_SCHEMA.COLUMNS 
        WHERE TABLE_SCHEMA = ? AND TABLE_NAME = ?
        ORDER BY ORDINAL_POSITION
    ");
    $stmt->execute([$database, $table]);
    $cols = [];
    foreach ($stmt->fetchAll(PDO::FETCH_ASSOC) as $row) {
        $cols[$row['col']] = [
            'type' => $row['type'],
            'nullable' => $row['nullable'],
            'default' => $row['default_val']
        ];
    }
    return $cols;
}

/**
 * Obter chave primária
 */
function getPrimaryKey($pdo, $database, $table) {
    $stmt = $pdo->prepare("
        SELECT COLUMN_NAME 
        FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE 
        WHERE TABLE_SCHEMA = ? AND TABLE_NAME = ? AND CONSTRAINT_NAME = 'PRIMARY'
        ORDER BY ORDINAL_POSITION
    ");
    $stmt->execute([$database, $table]);
    return $stmt->fetchAll(PDO::FETCH_COLUMN);
}

/**
 * Obter nomes de índices (excluindo PRIMARY)
 */
function getIndexes($pdo, $database, $table) {
    $stmt = $pdo->prepare("
        SELECT INDEX_NAME 
        FROM INFORMATION_SCHEMA.STATISTICS 
        WHERE TABLE_SCHEMA = ? AND TABLE_NAME = ? AND INDEX_NAME != 'PRIMARY'
        GROUP BY INDEX_NAME
        ORDER BY INDEX_NAME
    ");
    $stmt->execute([$database, $table]);
    return $stmt->fetchAll(PDO::FETCH_COLUMN);
}

/**
 * Contar registros em uma tabela
 */
function getRowCount($pdo, $database, $table) {
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM `$database`.`$table`");
    $stmt->execute();
    return $stmt->fetchColumn();
}
?>